<?php

// Use:
//   php -S localhost:8000
// To quickly test

// Requires curl
include 'Tulus.php';

const PAYMENT_ENDPOINT = "https://pay.tulus.my/";

// ----------------------------------------------------------------------------------------------------------------
function isSecure() {
    if (!empty($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] == 'https' || !empty($_SERVER['HTTP_X_FORWARDED_SSL']) && $_SERVER['HTTP_X_FORWARDED_SSL'] == 'on') {
        return true;
    }

    return
      (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
      || $_SERVER['SERVER_PORT'] == 443;
}
$scheme = isSecure() ? 'https' : 'http';

// The returnUrl here is for getting the result of the transaction.
// There are two ways for this to happen: "indirect" and "direct"
// In this example, it's processed in the
//
// "Indirect" is a request POSTed by the client's browser with after
// the completion of the request.
// "Direct" is a request POSTed by SnapNPay's server after the completion
// of the request.

// This parameter can be set during "dev" mode, but "prod" mode may
// impose more restrictions in terms of which addresses are valid.

// The data returned in $_POST looks like this:
//
/*
    Array
    (
        [status] => success
        [orderNo] => LP-0000179356e010000
        [refNo] => test5dbfdd8d9db55
        [amount] => 2.00
        [fpxTxnId] => 1911041613340758
        [extraData] => 881225-08-1234
    )
*/

// ----------------------------------------------------------------------------------------------------------------
// Payment Order API Example
// This is vastly simplified. Use and adapt for your own class
// Unless otherwise stated, the types in the following fields are specified in string, and is up to 255 bytes ASCII
// Use of UTF-8 values in the following may result in undefined behaviours.
$returnUrl = sprintf("%s://%s", $scheme, $_SERVER['HTTP_HOST']);
// This (receipt.php) demonstrates handling of customized receipts upon payment completion (either succeed or fail)
// via direct or indirect call backs. See example/receipt.php for more information.
$returnUrl .= "/sdk/example/receipt.php";
$returnUrl .= "?just=1&a=test";
$order = array(
    'agency'       => 'BERRYPAY',          // Enter your agency code here.
    'refNo'        => uniqid("test"),      // Generate a unique ID, prefixed with "test", but append some "extraData"
    'amount'       => "2.00",              // Amount in Ringgit
    'email'        => "sdk@snapnpay.my",   // Email address of payer to receive payment receipt
    'returnUrl'    => $returnUrl,          // callback url, change this to your form
    'utm_source'   => 'sdk',               // Optional, used for tracking (e.g. facebook, google, twitter, etc)
    'utm_medium'   => 'dev',               // Optional, used for tracking 
    'utm_campaign' => 'sdk_demo',          // Optional, used for tracking
    'utm_term'     => 'tulus sdk',         // Optional, used for tracking
    'utm_content'  => '',                  // Optional, used for tracking
    'pref_method'  => '',                  // Optional, "fpx", "mpgs"
    'pref_bank'    => '',                  // Optional.
    'extra_data'   => '',                  // Optional, used in lieu of "~" in refNo
);

// $extraData is extra, *optional* reference for helping the calling app, e.g. adding a MyKad reference.
// it can be anything alphanum, except "~"
$extraData = implode("~", array("881225-08-1234","12 \r\n Jalan Foo"));

$refNoDisplay = "Sample RefNo";

if ($extraData != "") {
    // We hitch on refNo to give this extradata
    $order['refNo'] = $order['refNo'] . "~" . $extraData;
}

if ($refNoDisplay != "") {
    // We hitch on refNo to add preferred display during checkout
    $order['refNo'] = $order['refNo'] . "~#" . $refNoDisplay;
}

// Default QR Code Generator for the SnapNPay App
// This is simplified structure of the QRCode generator. Use and adapt for your own class.
$qrcode = array(
    'agency'          => $order['agency'],
    'paymentCategory' => 'test',
    'refNo'           => uniqid("test"),
);
// Note the similarity of the qrcode structure with $order above, but:
// "email" is not used -- they will be filled in by the SnapNPay App
// "returnUrl" is optional

// There are two types of QRCode generation, secure ("signed/tamper resistant") and unsigned.
// You need to sign in (and get the secure token), to be able to generate the secured QR Code.
if (false) {
    // Example:
    $token = $snp->GetToken("myagency", "mypassword");
    for ($i = 0; $i < 10; $i++) {
        $qrcode['refNo'] = uniqid($i);
        $snp->GetSecureQRCodeUrl($qrcode); // Returns a URL that will return a PNG of the QR Code.
    }
}

// ----------------------------------------------------------------------------------------------------------------
// *Indirect* + *Direct* handling:
// Example of getting and processing the result from SnapNPay/FPX:
if (isset($_POST['status'])) {
    // This is a callback, ie, a return from processing the payment.
    $callback_status = $_POST;
} else {
    $callback_status = false;
}

// ----------------------------------------------------------------------------------------------------------------
// Payment Reports API Example
$params = array();
$authtoken = "";
$output = "";
$function = "";
if (($_POST['function'] ?? "") == "GetPaymentReport") {
    $function = "GetPaymentReport";
    $tulus = new Tulus\Tulus();

    // Tulus Auth Token has your access identity, scope and credential baked in.
    // This is no longer directly linked to any single agency or user.
    $authtoken = $_POST['authtoken'] ?? "";
    $tulus->SetAuthToken($authtoken);

    $params = array(
        'items' => 25, // Items per page
        'agency' => $_POST['agency'] ?? '',

    );
    $output = $tulus->GetPaymentReport('json', $params);
}

?>
<!DOCTYPE html>
<html lang="en">
    <head>
        <meta charset="UTF-8" />
        <meta name="viewport" content="width=device-width, initial-scale=1.0" />
        <meta http-equiv="X-UA-Compatible" content="ie=edge" />
        <title>Tulus/SnapNPay SDK Demo</title>
        <link rel="icon" type="image/png" href="https://assets.tulus.my/tulus-icon.png" sizes="32x32" />
        <link rel="stylesheet" href="https://assets.tulus.my/bootstrap-4.5/css/bootstrap.min.css"
            crossorigin="anonymous">
        <script src="https://code.jquery.com/jquery-3.5.1.slim.min.js"
            integrity="sha384-DfXdz2htPH0lsSSs5nCTpuj/zy4C+OGpamoFVy38MVBnE+IbbVYUew+OrCXaRkfj"
            crossorigin="anonymous"></script>
        <script src="https://assets.tulus.my/popperjs/popper.js"></script>
        <script src="https://assets.tulus.my/bootstrap-4.5/js/bootstrap.min.js"
            crossorigin="anonymous"></script>
        <script>
            function isValidEmail(email) {
                var regex = /^\w+([\.-]?\w+)*@\w+([\.-]?\w+)*(\.\w{2,3})+$/;
                return email.match(regex);
            }
            function checkValidEmail(email) {
                if (!isValidEmail(email)) {
                    alert("Please enter a valid email");
                }
            }
        </script>
    </head>
    <body>
        <div class="d-flex flex-column flex-md-row align-items-center p-3 px-md-4 mb-3 bg-white border-bottom box-shadow">
            <h5 class="my-0 mr-md-auto font-weight-normal">SnapNPay API - For both Tulus and SnapNPay</h5>
        </div>
        <div class="container p-0">
            <div>
                The API can be logically grouped in to 2:<br />
            </div>
            <ol>
                <li>Payment Orders</li>
                <li>Payment Reports</li>
            </ol>
            
            <div class="row">
                There are multiple ways to authenticate for the API, but the current more secure and flexible way
                is to specify a user and an API key.
            </div>
        </div>

<?php if ($output != ""): ?>
    <hr />
    <div class="container p-0">
        <h3>The result of <?= $function ?>:</h3>
        <pre><?= htmlspecialchars(json_encode(json_decode($output), JSON_PRETTY_PRINT)); ?></pre>
    </div>
<?php endif ?>
        <hr />
        <div class="container p-0">
            <h3>Payment Order</h3>
            <!-- Payment Example -->
            <div style="border: 2px solid; padding: 0.5em" class="container">
                Example Payment Order:
                <!-- API Key is optional, to sign and lock in the amount -->

                <pre>order = <?php print_r($order); ?></pre>

                <!-- ENDPOINT by default should be "https://pay.tulus.my/" for the most features and options -->
                <form method="post" id="payment" action="<?php print PAYMENT_ENDPOINT; ?>">
                    <div class="form-group p-0 m-0 row">
                        <label for="produat" class="col-sm-2 col-form-label"><b>Production / UAT</b></label>
                        <div class="col-sm-4">
                            <select class="form-control" id="produat">
                                <option>Production</option>
                                <option>UAT</option>
                                <option>UATOLD</option>
                                <option>Dev-FPX</option>
                                <option>Dev</option>
                                <option>LegacyFPX</option>
                            </select>
                        </div>
                        <div class="col-sm-6" >
                            <small class="form-text text-muted">Production - Live payments<br>
                            UAT - MEPS Dev Environment
                            </small>
                        </div>
                    </div>
                    <div class="form-group p-0 m-0 row">
                        <label for="refNo" class="col-sm-2 col-form-label"><b>Reference No</b>:<br /><code>refNo</code>
                        (Required)</label>
                        <div class="col-sm-4">
                            <input type="text" size=40 name="refNo" value="<?php echo htmlentities($order["refNo"]); ?>" class="form-control" id="refNo" aria-describedby="refnoHelp" />
                        </div>
                        <div class="col-sm-6">
                            <small id="refnoHelp" class="form-text text-muted">
                            Reference Number of this payment order.<br />
                            Used to uniquely identify this payment Order.
                            <br />
                            This would be generated by your system and may includue extra data by appending it with <code>~</code>
                            See this source code in the downloaded SDK for details.
                            </small>
                        </div>
                    </div>

                    <div class="form-group p-0 m-0 row">
                        <label for="amount" class="col-sm-2 col-form-label"><b>Amount</b>:<br /><code>amount</code>
                            (Required)</label>
                        <div class="col-sm-4">
                            <input type="text" size=40 name="amount" value="<?php echo htmlentities($order["amount"]??""); ?>" class="form-control" id="amount" aria-describedby="amountHelp" />
                        </div>
                        <div class="col-sm-6">
                            <small id="amountHelp" class="form-text text-muted">
                            The Amount, in Ringgit Malaysia. e.g. <code>12.20</code>
                            Bear in mind that the amount for FPX B2B is RM 2.00 after charges, and FPX B2C is RM 1.00 after charges.
                            Payment methods may be disabled and hidden if the minimum is not met.
                            </small>
                        </div>
                    </div>

                    <div class="form-group p-0 m-0 row">
                        <label for="email" class="col-sm-2 col-form-label"><b>Email</b>:<br /><code>email</code>
                        (Required)</label>
                        <div class="col-sm-4">
                            <input type="text" size=40 name="email" value="<?php echo htmlentities($order["email"]??""); ?>" onchange="checkValidEmail(this.value)" class="form-control" id="email" aria-describedby="emailHelp" />
                        </div>
                    
                        <div class="col-sm-6">
                            <small id="emailHelp" class="form-text text-muted">
                            Valid Email address. Receipt will be sent here.
                            </small>
                        </div>
                    </div>

                    <div class="form-group p-0 m-0 row">
                        <label for="agency" class="col-sm-2 col-form-label"><b>Agency Code</b>:<br /><code>agency</code>
                        (Required)</label>
                        <div class="col-sm-4">
                            <input type="text" size=40 name="agency" value="<?php echo htmlentities($order["agency"]??""); ?>" class="form-control" id="agency" aria-describedby="agencyHelp" />
                        </div>
                    
                        <div class="col-sm-6">
                            <small id="agencyHelp" class="form-text text-muted">
                            Valid Agency Code.
                            </small>
                        </div>
                    </div>
                    
                    <div class="form-group p-0 m-0 row">
                        <label for="returnUrl" class="col-sm-2 col-form-label"><b>Return URL</b>:<br /><code>returnUrl</code>
                        (to be hidden)</label>
                        <div class="col-sm-4">
                            <input type="text" size=40 name="returnUrl" value="<?php echo htmlentities($order["returnUrl"]??""); ?>" class="form-control" id="returnUrl" aria-describedby="returnUrlHelp" />
                        </div>
                    
                        <div class="col-sm-6">
                            <small id="returnUrlHelp" class="form-text text-muted">
                            Return URL. Redirect here when all done.
                            </small>
                        </div>
                    </div>

                    <div class="form-group p-0 m-0 row">
                        <label for="utm_source" class="col-sm-2 col-form-label"><b>UTM Source</b>:<br /><code>utm_source</code>
                        </label>
                        <div class="col-sm-4">
                            <input type="text" size=40 name="utm_source" value="<?php echo htmlentities($order["utm_source"]??""); ?>" class="form-control" id="utm_source" aria-describedby="returnUrlHelp" />
                        </div>
                    
                        <div class="col-sm-6">
                            <small id="utm_sourceHelp" class="form-text text-muted">
                            Identifies which site sent the traffic, and is a required parameter. e.g. (<code>instagram</code>)
                            </small>
                        </div>
                    </div>
                    
                    <div class="form-group p-0 m-0 row">
                        <label for="utm_medium" class="col-sm-2 col-form-label"><b>UTM Medium</b>:<br /><code>utm_medium</code>
                        </label>
                        <div class="col-sm-4">
                            <input type="text" size=40 name="utm_medium" value="<?php echo htmlentities($order["utm_medium"]??""); ?>" class="form-control" id="utm_medium" aria-describedby="returnUrlHelp" />
                        </div>
                    
                        <div class="col-sm-6">
                            <small id="utm_mediumHelp" class="form-text text-muted">
                            Identifies what type of link was used, such as cost per click or email. e.g. (<code>email</code>)
                            </small>
                        </div>
                    </div>

                    <div class="form-group p-0 m-0 row">
                        <label for="utm_campaign" class="col-sm-2 col-form-label"><b>UTM Campaign</b>:<br /><code>utm_campaign</code>
                        </label>
                        <div class="col-sm-4">
                            <input type="text" size=40 name="utm_campaign" value="<?php echo htmlentities($order["utm_campaign"]??""); ?>" class="form-control" id="utm_campaign" aria-describedby="utm_campaignHelp" />
                        </div>
                    
                        <div class="col-sm-6">
                            <small id="utm_campaignHelp" class="form-text text-muted">
                            Identifies a specific product promotion or strategic campaign. e.g. (<code>spring_sale</code>)
                            </small>
                        </div>
                    </div>

                    <div class="form-group p-0 m-0 row">
                        <label for="utm_term" class="col-sm-2 col-form-label"><b>UTM Term</b>:<br /><code>utm_term</code>
                        </label>
                        <div class="col-sm-4">
                            <input type="text" size=40 name="utm_term" value="<?php echo htmlentities($order["utm_term"]??""); ?>" class="form-control" id="utm_term" aria-describedby="utm_termHelp" />
                        </div>
                    
                        <div class="col-sm-6">
                            <small id="utm_termHelp" class="form-text text-muted">
                            Identifies search terms. e.g. (<code>pay zakat</code>)
                            </small>
                        </div>
                    </div>

                    <div class="form-group p-0 m-0 row">
                        <label for="utm_content" class="col-sm-2 col-form-label"><b>UTM Content</b>:<br /><code>utm_content</code>
                        </label>
                        <div class="col-sm-4">
                            <input type="text" size=40 name="utm_content" value="<?php echo htmlentities($order["utm_content"]??""); ?>" class="form-control" id="utm_content" aria-describedby="utm_contentHelp" />
                        </div>
                    
                        <div class="col-sm-6">
                            <small id="utm_contentHelp" class="form-text text-muted">
                            Identifies what specifically was clicked to bring the user to the site, such as a banner ad or a text link. It is often used for A/B testing. e.g. (<code>logolink</code>)
                            </small>
                        </div>
                    </div>

                    <div class="form-group p-0 m-0 row">
                        <label for="pref_method" class="col-sm-2 col-form-label"><b>Preferred Payment Method</b>:<br /><code>pref_method</code>
                        </label>
                        <div class="col-sm-4">
                            <input type="text" size=40 name="pref_method" value="<?php echo htmlentities($order["pref_method"]??""); ?>" class="form-control" id="pref_method" aria-describedby="pref_srcHelp" />
                        </div>
                    
                        <div class="col-sm-6">
                            <small id="pref_srcHelp" class="form-text text-muted">
                        [ALPHA] Preferred Payment Method (<code>fpx/b2b1</code> for B2B1 FPX)
                            </small>
                        </div>
                    </div>

                    <div class="form-group p-0 m-0 row">
                        <label for="pref_bank" class="col-sm-2 col-form-label"><b>Preferred Bank</b>:<br /><code>pref_bank</code>
                        </label>
                        <div class="col-sm-4">
                            <input type="text" size=40 name="pref_bank" value="<?php echo htmlentities($order["pref_bank"]??""); ?>" class="form-control" id="pref_bank" aria-describedby="pref_bankHelp" />
                        </div>
                    
                        <div class="col-sm-6">
                            <small id="pref_bankHelp" class="form-text text-muted">
                        [ALPHA] Preferred Bank e.g. (<code>ABB0235</code> for Affin Bank B2B(see <code>/api/v1/fpx/banklist/b2b</code>))
                            </small>
                        </div>
                    </div>

                    <div class="form-group p-0 m-0 row">
                        <label for="pref_bank" class="col-sm-2 col-form-label"><b>Preferred Bank</b>:<br /><code>pref_bank</code>
                        </label>
                        <div class="col-sm-4">
                            <input type="text" size=40 name="pref_bank" value="<?php echo htmlentities($order["pref_bank"]??""); ?>" class="form-control" id="pref_bank" aria-describedby="pref_bankHelp" />
                        </div>
                    
                        <div class="col-sm-6">
                            <small id="pref_bankHelp" class="form-text text-muted">
                        [ALPHA] Preferred Bank e.g. (<code>ABB0235</code> for Affin Bank B2B(see <code>/api/v1/fpx/banklist/b2b</code>))
                            </small>
                        </div>
                    </div>

                    <div class="form-group p-0 m-0 row">
                        <label for="debug" class="col-sm-2 col-form-label"><b>Debug string</b>:<br /><code>debug</code>
                        </label>
                        <div class="col-sm-4">
                            <input type="text" size=40 name="debug" value="<?php echo htmlentities($order["debug"]??""); ?>" class="form-control" id="debug" aria-describedby="debugHelp" />
                        </div>
                    
                        <div class="col-sm-6">
                            <small id="debugHelp" class="form-text text-muted">
                                Internal debug use
                            </small>
                        </div>
                    </div>
                                        
                    <div class="form-group p-0 m-0 row">
                        <div class="col-sm-2"></div>
                        <div class="col-sm-4">
                            <button class="btn btn-primary" type="submit">Submit</button>
                        </div>
                        <div class="col-sm-6">
                            POST to <code><?php print PAYMENT_ENDPOINT; ?></code>
                        </div>
                    </div>
                </form>
            </div>
        </div>
        <!-- End payment Example -->


        <hr />
        <div class="container p-0">
            <h3>Payment Reports</h3>
        </div>

        <div style="border: 2px solid; padding: 0.5em" class="container">
            <form action="index.php" method="POST">
                <input type="hidden" name="function" value="GetPaymentReport" />
                <h5>GetPaymentReport</h5>
                <div class="form-group p-0 m-0 row">
                    <label for="authtoken" class="col-sm-2 col-form-label"><b>Auth Token</b>:<br /><code>Bearer</code>
                    (Required)</label>
                    <div class="col-sm-4">
                        <input type="text" size=40 name="authtoken" value="<?php echo htmlentities($authtoken); ?>" class="form-control" id="refNo" aria-describedby="authtokenHelp" />
                    </div>
                    <div class="col-sm-6">
                        <small id="authtokenHelp" class="form-text text-muted">
                        You should obtained the AuthToken
                        by logging into the Merchant dashboard at https://reports.tulus.my/
                        and getting the AuthToken copied to clipboard from the top right menu,
                        "Copy AuthToken to Clipboard"
                        </small>
                    </div>
                </div>

                <div class="form-group p-0 m-0 row">
                    <label for="agency" class="col-sm-2 col-form-label"><b>Agency</b>:<br /><code>agency</code>
                    (Required)</label>
                    <div class="col-sm-4">
                        <input type="text" size=40 name="agency" value="<?php echo htmlentities($params['agency'] ?? ''); ?>" class="form-control" id="agency" aria-describedby="agencyHelp" />
                    </div>
                    <div class="col-sm-6">
                        <small id="agencyHelp" class="form-text text-muted">
                        Code of the agency.
                        </small>
                    </div>
                </div>

                <div class="form-group p-0 m-0 row">
                    <div class="col-sm-2"></div>
                    <div class="col-sm-10">
                        <button class="btn btn-primary" type="submit">GetPaymentReport</button>
                    </div>
                </div>
            </form>
        </div>

        <hr />
        <div class="container p-0">
        <!-- spacer -->
        </div>
    </div>

    <script>
        (function() {
            function fixurl() {
                let v = $("#produat").val();
                if (v == "UATOLD") {
                    $("#payment").attr("action", "https://dev.snapnpay.co/payments/api");
                } else if (v == "Production") {
                    $("#payment").attr("action", "https://pay.tulus.my/");
                } else if (v == "UAT") {
                    $("#payment").attr("action", "https://paydev.tulus.my/");
                } else if (v == "Dev-FPX") {
                    $("#payment").attr("action", "http://localhost:3001/payments/api");
                } else if (v == "LegacyFPX") {
                    var url = "/payments/mobile/checkout/";
                    url += encodeURIComponent($("#refNo").val()) + "/";
                    url += encodeURIComponent($("#amount").val()) + "/";
                    url += encodeURIComponent($("#agency").val());
                    $("#payment").attr("action", url);
                } else if (v == "Dev") {
                    // FPXPAY
                    $("#payment").attr("action", "http://localhost:9998/");
                }
            }
            fixurl(); // Call initially to set the correct action URL
            $("#produat").on("change", fixurl);
            $("#refNo").on("change", fixurl);
            $("#amount").on("change", fixurl);
            $("#agency").on("change", fixurl);
        })();
    </script>
    </body>
</html>